# -*- coding: utf-8 -*-
"""
ユーティリティクラス（ロガー管理と通知サービス）
"""

import logging
from datetime import datetime
from pathlib import Path
from typing import Optional
from plyer import notification
from config import Config


class LoggerManager:
    """ロガー管理クラス"""
    
    def __init__(self, logs_dir: Path):
        """ロガーマネージャーの初期化"""
        self.logs_dir = logs_dir
        self.logger = self._create_logger()
        self.current_log_date = datetime.now().strftime(Config.DATE_FORMAT)
    
    def _create_logger(self) -> logging.Logger:
        """ロガーを作成"""
        logger = logging.getLogger('ccresultss')
        logger.setLevel(logging.INFO)
        logger.handlers.clear()
        logger.propagate = False  # 親ロガーへの伝播を無効化（重複出力を防ぐ）
        
        # フォーマッター
        formatter = logging.Formatter(Config.LOG_FORMAT, datefmt=Config.LOG_TIME_FORMAT)
        
        # ファイルハンドラー
        log_file = self._get_log_file_path()
        file_handler = logging.FileHandler(log_file, mode='a', encoding='utf-8')
        file_handler.setLevel(logging.INFO)
        file_handler.setFormatter(formatter)
        logger.addHandler(file_handler)
        
        # コンソールハンドラー（標準出力）
        console_handler = logging.StreamHandler()
        console_handler.setLevel(logging.INFO)
        console_handler.setFormatter(formatter)
        logger.addHandler(console_handler)
        
        return logger
    
    def _get_log_file_path(self) -> Path:
        """現在のログファイルパスを取得"""
        date_str = datetime.now().strftime(Config.DATE_FORMAT)
        return self.logs_dir / Config.LOG_FILE_FORMAT.format(date=date_str)
    
    def check_rotation(self) -> None:
        """ログローテーションの確認
        
        日付が変わった場合、新しいログファイルに切り替えます。
        """
        current_date = datetime.now().strftime(Config.DATE_FORMAT)
        if current_date != self.current_log_date:
            self.current_log_date = current_date
            self.logger = self._create_logger()


class NotificationService:
    """通知サービスクラス"""
    
    @staticmethod
    def send(message: str, logger: Optional[logging.Logger] = None) -> None:
        """デスクトップ通知を送信
        
        非同期処理でパフォーマンスに影響を与えません。
        注意: フルスクリーンアプリケーションが前面にある場合、Windowsの仕様により通知が表示されないことがあります。
        通知はWindowsのアクションセンター（通知センター）に記録されます。
        
        Args:
            message: 通知メッセージ
            logger: ロガー（オプション）
        """
        try:
            notification.notify(
                title=Config.NOTIFICATION_TITLE,
                message=message,
                timeout=Config.NOTIFICATION_TIMEOUT
            )
            if logger:
                logger.info(f"通知を送信しました: {message}（アクションセンターで確認できます）")
        except Exception as e:
            if logger:
                logger.error(f"通知送信エラー: {e}")

