# -*- coding: utf-8 -*-
"""
テンプレート画像管理クラスとテンプレートマッチングクラス
"""

import re
import time
import logging
from pathlib import Path
from typing import Optional, List, Tuple
import cv2
import numpy as np
from config import Config


class TemplateImage:
    """テンプレート画像管理クラス"""
    
    def __init__(self, template_path: Path, logger: logging.Logger):
        """テンプレート画像の初期化"""
        self.template_path = template_path
        self.logger = logger
        self.image: Optional[np.ndarray] = None
        self.load()
    
    def load(self) -> None:
        """テンプレート画像を読み込み"""
        if not self.template_path.exists():
            raise FileNotFoundError(f"テンプレート画像が見つかりません: {self.template_path}")
        
        self._validate_file()
        self._load_image_data()
    
    def _validate_file(self) -> None:
        """ファイルの妥当性を検証"""
        file_stat = self.template_path.stat()
        if file_stat.st_size == 0:
            raise ValueError(f"テンプレート画像ファイルが空です: {self.template_path}")
    
    def _load_image_data(self) -> None:
        """画像データを読み込む（日本語パス対応）"""
        try:
            image_data = self._read_image_file()
            self.image = self._decode_image(image_data)
            if self.image is None:
                raise ValueError(
                    f"テンプレート画像のデコードに失敗しました: {self.template_path}\n"
                    f"ファイルが破損している可能性があります。\n"
                    f"OneDriveを使用している場合、ファイルを右クリックして「常にこのデバイスで使用可能にする」を選択してください。"
                )
        except PermissionError as e:
            raise RuntimeError(f"テンプレート画像ファイルへのアクセス権限がありません: {self.template_path} ({e})") from e
        except OSError as e:
            raise RuntimeError(
                f"テンプレート画像ファイルの読み取りエラー: {self.template_path} ({e})\n"
                f"OneDriveの「オンラインのみ」ファイルの可能性があります。\n"
                f"ファイルを右クリックして「常にこのデバイスで使用可能にする」を選択してください。"
            ) from e
    
    def _read_image_file(self) -> bytes:
        """画像ファイルを読み込む
        
        Returns:
            画像ファイルのバイトデータ
            
        Raises:
            ValueError: ファイルが不完全、またはPNG形式ではない場合
        """
        with open(self.template_path, 'rb') as f:
            header = f.read(8)
            if len(header) < 8:
                raise ValueError(
                    f"テンプレート画像ファイルが不完全です: {self.template_path}\n"
                    f"OneDriveの「オンラインのみ」ファイルの可能性があります。\n"
                    f"ファイルを右クリックして「常にこのデバイスで使用可能にする」を選択してください。"
                )
            
            if header[:4] != b'\x89PNG':
                raise ValueError(f"テンプレート画像ファイルがPNG形式ではありません: {self.template_path}")
            
            f.seek(0)
            return f.read()
    
    def _decode_image(self, image_data: bytes) -> Optional[np.ndarray]:
        """画像データをデコード
        
        Args:
            image_data: 画像ファイルのバイトデータ
            
        Returns:
            デコードされた画像データ（BGR形式）、失敗時はNone
        """
        image_array = np.frombuffer(image_data, np.uint8)
        return cv2.imdecode(image_array, cv2.IMREAD_COLOR)
    
    def is_loaded(self) -> bool:
        """テンプレート画像が読み込まれているか"""
        return self.image is not None


class TemplateMatcher:
    """テンプレートマッチングクラス（複数テンプレート対応）"""
    
    def __init__(self, logger: logging.Logger, template_groups: List[List[TemplateImage]], 
                 scales_settings: List[List[int]]):
        """テンプレートマッチャーの初期化
        
        Args:
            logger: ロガー
            template_groups: テンプレートグループのリスト（各グループはテンプレート画像のリスト）
            scales_settings: 各テンプレートグループで使用するスケールのリスト
        """
        self.template_groups = template_groups
        self.logger = logger
        self.scales_settings = scales_settings
    
    def _filter_templates_by_scale(self, template_group: List[TemplateImage], target_scales: List[int]) -> List[TemplateImage]:
        """テンプレートをスケールでフィルタリング"""
        if not target_scales:
            return template_group
        
        templates_to_check = []
        for template in template_group:
            if not template.is_loaded():
                continue
            # ファイル名からスケールを抽出（例: template1_100.png -> 100）
            scale_match = re.search(r'_(\d+)$', template.template_path.stem)
            if scale_match:
                scale = int(scale_match.group(1))
                if scale in target_scales:
                    templates_to_check.append(template)
            else:
                # パターンに合わない場合は含める（フォールバック）
                templates_to_check.append(template)
        return templates_to_check
    
    def detect_all(self, image: np.ndarray) -> Tuple[List[bool], float]:
        """すべてのテンプレート画像の検出状態を返す（各テンプレートグループごとにTrue/False）と処理時間（秒）"""
        if image is None:
            # グループ数に応じてFalseのリストを返す
            return [False] * len(self.template_groups), 0.0
        
        start_time = time.perf_counter()
        img_height, img_width = image.shape[:2]
        
        results = []
        
        # 各テンプレートグループについて検出
        for group_index, template_group in enumerate(self.template_groups):
            best_match = 0.0
            
            # このグループで使用するスケールを取得
            target_scales = (
                self.scales_settings[group_index]
                if group_index < len(self.scales_settings)
                else []
            )
            
            # 対象スケールのみを検出対象とする
            templates_to_check = self._filter_templates_by_scale(template_group, target_scales)
            
            # 早期終了：十分高い一致度が見つかったら処理を終了
            for template in templates_to_check:
                try:
                    template_height, template_width = template.image.shape[:2]
                    
                    # テンプレート画像が元画像より大きい場合はスキップ
                    if template_height > img_height or template_width > img_width:
                        continue
                    
                    result = cv2.matchTemplate(image, template.image, cv2.TM_CCOEFF_NORMED)
                    _, max_val, _, _ = cv2.minMaxLoc(result)
                    
                    if max_val > best_match:
                        best_match = max_val
                    
                    # 早期終了：しきい値を超える一致度が見つかったら、このグループの検出を終了
                    if best_match >= Config.TEMPLATE_MATCH_THRESHOLD:
                        break
                except Exception as e:
                    self.logger.error(f"テンプレート検出エラー ({template.template_path}): {e}")
                    continue
            
            results.append(best_match >= Config.TEMPLATE_MATCH_THRESHOLD)
        
        elapsed_time = time.perf_counter() - start_time
        return results, elapsed_time

