# -*- coding: utf-8 -*-
"""
CCResultSS
特定のゲームウィンドウを監視し、特定の画像検知時に自動撮影するアプリケーション

Copyright (c) 2025 CCRank
Licensed under the MIT License
See LICENSE file for details
"""

import sys
import logging
import traceback
from datetime import datetime
from game_window_monitor import GameWindowMonitor
from path_manager import PathManager
from config import Config


def _write_error_log(error_msg: str, traceback_str: str) -> None:
    """エラーログをファイルに書き込む（失敗しても例外を投げない）
    
    Args:
        error_msg: エラーメッセージ
        traceback_str: トレースバック文字列
    """
    try:
        paths = PathManager()
        paths.ensure_directories()
        timestamp = datetime.now().strftime(Config.DATE_FORMAT)
        log_file = paths.logs_dir / f"ccresultss_error_{timestamp}.log"
        logging.basicConfig(
            level=logging.ERROR,
            format='[%(asctime)s] %(message)s',
            datefmt='%H:%M:%S',
            handlers=[logging.FileHandler(log_file, encoding='utf-8')]
        )
        logger = logging.getLogger('ccresultss_error')
        logger.error(error_msg)
        logger.error(traceback_str)
    except Exception:
        # ログファイルへの書き込みも失敗した場合は無視
        pass


def main() -> None:
    """メイン関数"""
    try:
        monitor = GameWindowMonitor()
        monitor.run()
    except KeyboardInterrupt:
        # Ctrl+Cで終了（game_window_monitor.pyで処理されるが、念のため）
        print("\nアプリケーションを終了します...")
        sys.exit(0)
    except Exception as e:
        error_msg = f"エラーが発生しました: {e}"
        print(error_msg)
        traceback.print_exc()
        _write_error_log(error_msg, traceback.format_exc())
        
        sys.exit(1)


if __name__ == '__main__':
    main()
