# -*- coding: utf-8 -*-
"""
ゲームウィンドウ監視クラス
"""

import re
import time
from pathlib import Path
from typing import Optional, Tuple, List, Dict
import numpy as np
import pygetwindow as gw
from config import Config, ConfigManager
from path_manager import PathManager
from utils import LoggerManager, NotificationService
from template import TemplateImage, TemplateMatcher
from capture import WindowCapturer, ScreenshotManager
from capture_winrt import initialize_capture_manager, shutdown_capture_manager


class GameWindowMonitor:
    """ゲームウィンドウを監視して自動撮影を行うクラス"""
    
    def __init__(self):
        """初期化処理"""
        # パス管理
        self.paths = PathManager()
        self.paths.ensure_directories()
        
        # 設定管理
        config_manager = ConfigManager(self.paths.config_path)
        self.target_window_title = config_manager.load_window_title()
        image_format = config_manager.load_image_format()
        
        # ロガー管理
        self.logger_manager = LoggerManager(self.paths.logs_dir)
        self.logger = self.logger_manager.logger
        
        # テンプレート画像管理（プログラム内で固定）
        template_groups_dict, failed_files = self._load_template_images(config_manager)
        template_groups, template_names, scales_settings = self._organize_template_groups(
            template_groups_dict, config_manager
        )
        
        if not template_groups:
            error_msg = "有効なテンプレートグループが1つも見つかりませんでした。"
            self.logger.error(error_msg)
            raise ValueError(error_msg) from None
        
        self.template_names = template_names
        self.template_matcher = TemplateMatcher(self.logger, template_groups, scales_settings)
        
        # 待ち時間設定を読み込み
        self.delays = self._load_delays(config_manager, template_names)
        
        # 撮影回数設定を読み込み（エラー時は起動を停止）
        self.shoot_counts = self._load_shoot_counts(config_manager, template_names)
        
        # キャプチャ・保存管理
        self.window_capturer = WindowCapturer(self.logger)
        self.screenshot_manager = ScreenshotManager(self.paths.screenshots_dir, self.logger, image_format)
        
        # 監視状態
        self.last_detection_time = 0.0
        num_template_groups = len(template_groups)
        self.was_detected = [False] * num_template_groups
        self.undetected_count = [0] * num_template_groups
        
        # ウィンドウ検索結果のキャッシュ
        self.cached_window = None
        self.last_window_search_time = 0.0
        
        # ログローテーション確認のキャッシュ
        self.last_log_rotation_check_time = 0.0
        
        # 起動時のログ出力
        self.logger.info("=" * 50)
        self.logger.info(f"ccresultss 起動")
        self.logger.info(f"ターゲットウィンドウ: {self.target_window_title}")
        self.logger.info(f"スクリーンショット保存先: {self.paths.screenshots_dir}")
        self.logger.info("=" * 50)
        
        # 起動時にウィンドウが見つかるか確認
        if self._find_window() is None:
            error_msg = (
                f"ターゲットウィンドウ '{self.target_window_title}' が見つかりませんでした。\n"
                "ゲームを起動してからアプリケーションを実行してください。"
            )
            self.logger.error(error_msg)
            raise RuntimeError(error_msg) from None
        
        # Windows Capture Managerを初期化（WinRT使用時）
        # 初期化に失敗してもアプリは継続（PrintWindow/mssにフォールバック）
        initialize_capture_manager(self.target_window_title, self.logger)
    
    def _organize_template_groups(
        self,
        template_groups_dict: Dict[str, List[TemplateImage]],
        config_manager: ConfigManager
    ) -> Tuple[List[List[TemplateImage]], List[str], List[List[int]]]:
        """テンプレートグループを整理して、グループ、名前、スケール設定のリストを返す"""
        template_groups = []
        template_names = []
        scales_settings = []
        
        for index, base_name in enumerate(Config.TEMPLATE_BASE_NAMES):
            if base_name not in template_groups_dict:
                continue
            
            template_groups.append(template_groups_dict[base_name])
            
            # スクリーンショット名を取得
            screenshot_name = (
                Config.TEMPLATE_SCREENSHOT_NAMES[index]
                if index < len(Config.TEMPLATE_SCREENSHOT_NAMES)
                else base_name
            )
            template_names.append(screenshot_name)
            
            # 基準サイズから使用するスケールを計算
            base_scale = config_manager.load_template_base_scale(screenshot_name)
            scales = config_manager.calculate_optimal_scales(base_scale)
            scales_settings.append(scales)
        
        return template_groups, template_names, scales_settings
    
    def _load_template_images(
        self,
        config_manager: ConfigManager
    ) -> Tuple[Dict[str, List[TemplateImage]], List[str]]:
        """テンプレート画像を読み込んでグループ化"""
        template_files = config_manager.load_template_files()
        template_groups_dict: Dict[str, List[TemplateImage]] = {}
        failed_files = []
        
        for template_file in template_files:
            template_path = self.paths.templates_dir / template_file
            try:
                template = TemplateImage(template_path, self.logger)
                base_name = self._extract_base_name(template_file)
                template_groups_dict.setdefault(base_name, []).append(template)
            except (FileNotFoundError, ValueError, RuntimeError) as e:
                failed_files.append(template_file)
                self.logger.warning(f"テンプレート画像の読み込みをスキップしました: {template_file} (パス: {template_path}) - {e}")
        
        if not template_groups_dict:
            self._raise_template_load_error(template_files, failed_files)
        
        return template_groups_dict, failed_files
    
    def _raise_template_load_error(self, template_files: List[str], failed_files: List[str]) -> None:
        """テンプレート読み込みエラーを発生させる"""
        templates_dir_str = str(self.paths.templates_dir)
        error_msg = (
            f"有効なテンプレート画像が1つも読み込めませんでした。\n"
            f"テンプレートディレクトリ: {templates_dir_str}\n"
            f"\n"
            f"OneDriveを使用している場合:\n"
            f"  1. テンプレートフォルダを右クリック\n"
            f"  2. 「常にこのデバイスで使用可能にする」を選択\n"
            f"  3. ファイルがダウンロードされるまで待ってから再度実行\n"
            f"\n"
            f"または、プロジェクトをOneDrive以外のローカルディレクトリに移動してください。"
        )
        self.logger.error(error_msg)
        self.logger.error(f"テンプレートディレクトリの存在確認: {self.paths.templates_dir.exists()}")
        
        if self.paths.templates_dir.exists():
            self._log_template_directory_details(template_files, failed_files)
        
        raise FileNotFoundError(error_msg) from None
    
    def _log_template_directory_details(self, template_files: List[str], failed_files: List[str]) -> None:
        """テンプレートディレクトリの詳細をログに記録"""
        try:
            files = list(self.paths.templates_dir.iterdir())
            self.logger.error(f"ディレクトリ内のファイル数: {len(files)}")
            if files:
                self.logger.error(f"最初の5つのファイル: {[f.name for f in files[:5]]}")
                self.logger.error(f"期待されるファイル数: {len(template_files)}個")
                if failed_files:
                    self.logger.error(f"読み込み失敗したファイル数: {len(failed_files)}個")
                    self.logger.error(f"読み込み失敗したファイル（最初の5つ）: {failed_files[:5]}")
        except Exception as e:
            self.logger.error(f"ディレクトリ内容の取得に失敗: {e}")
    
    def _extract_base_name(self, template_file: str) -> str:
        """テンプレートファイル名からベース名を抽出"""
        base_name_match = re.match(r'^(.+?)_\d+\.png$', template_file)
        if base_name_match:
            return base_name_match.group(1)
        return template_file.rsplit('_', 1)[0].rsplit('.', 1)[0]
    
    def _load_delays(self, config_manager: ConfigManager, template_names: List[str]) -> Dict[str, float]:
        """待ち時間設定を読み込み
        
        Args:
            config_manager: 設定マネージャー
            template_names: テンプレート名のリスト
            
        Returns:
            テンプレート名をキー、待ち時間を値とする辞書
        """
        delays = {}
        for template_name in template_names:
            delay = config_manager.load_delay(template_name)
            delays[template_name] = delay
            if delay > 0:
                self.logger.info(f"{template_name} の待ち時間: {delay}秒")
        return delays
    
    def _load_shoot_counts(self, config_manager: ConfigManager, template_names: List[str]) -> Dict[str, int]:
        """撮影回数設定を読み込み（エラー時は起動を停止）
        
        Args:
            config_manager: 設定マネージャー
            template_names: テンプレート名のリスト
            
        Returns:
            テンプレート名をキー、撮影回数を値とする辞書
            
        Raises:
            RuntimeError: 設定値が無効な場合
        """
        shoot_counts = {}
        for template_name in template_names:
            try:
                shoot_count = config_manager.load_shoot_count(template_name)
                shoot_counts[template_name] = shoot_count
                if shoot_count > 1:
                    self.logger.info(f"{template_name} の撮影回数: {shoot_count}回")
            except ValueError as e:
                error_msg = str(e)
                self.logger.error(error_msg)
                self.logger.error(
                    f"設定ファイル ({self.paths.config_path}) を確認し、"
                    f"正しい値を設定してください。"
                )
                raise RuntimeError(error_msg) from None
        return shoot_counts
    
    def _find_window(self) -> Optional[gw.Window]:
        """ターゲットウィンドウを検索（キャッシュ機能付き）
        
        Returns:
            見つかったウィンドウ、見つからない場合はNone
        """
        current_time = time.time()
        
        # キャッシュが有効で、キャッシュ時間内の場合はキャッシュを返す
        if (self.cached_window is not None and 
            current_time - self.last_window_search_time < Config.WINDOW_CACHE_INTERVAL):
            return self.cached_window
        
        # ウィンドウを検索
        try:
            windows = gw.getWindowsWithTitle(self.target_window_title)
            if not windows:
                self.cached_window = None
                self.last_window_search_time = current_time
                return None
            
            # 完全一致または部分一致するウィンドウを探す
            for window in windows:
                if self.target_window_title in window.title:
                    self.cached_window = window
                    self.last_window_search_time = current_time
                    return window
            
            # マッチしなかった場合は最初のウィンドウを返す（フォールバック）
            if windows:
                self.logger.warning(
                    f"完全一致するウィンドウが見つかりませんでした。最初のウィンドウ '{windows[0].title}' を使用します。"
                )
            self.cached_window = windows[0] if windows else None
            self.last_window_search_time = current_time
            return self.cached_window
        except Exception as e:
            self.logger.error(f"ウィンドウ検索エラー: {e}")
            self.cached_window = None
            self.last_window_search_time = current_time
            return None
    
    def _should_take_screenshot(self, detection_results: List[bool], num_template_groups: int) -> bool:
        """撮影が必要かどうかを判定
        
        Args:
            detection_results: 各テンプレートグループの検出結果のリスト
            num_template_groups: テンプレートグループ数
            
        Returns:
            撮影が必要な場合True、そうでない場合False
        """
        # 新しく検出されたテンプレートグループがあるか確認
        for i in range(num_template_groups):
            is_newly_detected = detection_results[i] and not self.was_detected[i]
            if is_newly_detected:
                # 初回検出、または他のグループが既に検出されていた場合は撮影
                is_any_previously_detected = any(self.was_detected[j] for j in range(num_template_groups) if j != i)
                is_first_detection = not any(self.was_detected)
                if is_any_previously_detected or is_first_detection:
                    return True
        return False
    
    def _update_detection_state(self, detection_results: List[bool], num_template_groups: int) -> None:
        """検出状態を更新"""
        for i in range(num_template_groups):
            is_detected = detection_results[i]
            if is_detected:
                self.undetected_count[i] = 0
                self.was_detected[i] = True
            else:
                if self.was_detected[i]:
                    self.undetected_count[i] = 1
                elif self.undetected_count[i] > 0:
                    self.undetected_count[i] += 1
                self.was_detected[i] = False
    
    def _find_detected_template_index(self, detection_results: List[bool], num_template_groups: int) -> int:
        """検出されたテンプレートのインデックスを取得（新しく検出されたテンプレートを優先）"""
        # 新しく検出されたテンプレートを優先
        for i in range(num_template_groups):
            if detection_results[i] and not self.was_detected[i]:
                return i
        # 新しく検出されたテンプレートがない場合、検出されているテンプレートを探す
        for i in range(num_template_groups):
            if detection_results[i]:
                return i
        return 0
    
    def _should_check_now(self, current_time: float) -> bool:
        """現在監視を実行すべきかチェック"""
        return current_time - self.last_detection_time >= Config.DETECTION_INTERVAL
    
    def _process_window_detection(self, window: gw.Window, current_time: float) -> None:
        """ウィンドウ検出時の処理（各テンプレートを独立して管理）"""
        screenshot, capture_time, _ = self.window_capturer.capture_window(window)
        if screenshot is None:
            return
        
        detection_results, detection_time = self.template_matcher.detect_all(screenshot)
        num_template_groups = len(detection_results)
        
        should_take_screenshot = self._should_take_screenshot(detection_results, num_template_groups)
        self._update_detection_state(detection_results, num_template_groups)
        
        if should_take_screenshot:
            self._take_screenshot(window, screenshot, detection_results, num_template_groups, capture_time, detection_time)
    
    def _take_screenshot(
        self,
        window: gw.Window,
        screenshot: np.ndarray,
        detection_results: List[bool],
        num_template_groups: int,
        detection_capture_time: float,
        detection_time: float
    ) -> None:
        """スクリーンショットを撮影（複数回撮影対応）"""
        detected_template_index = self._find_detected_template_index(detection_results, num_template_groups)
        suffix = (
            self.template_names[detected_template_index]
            if detected_template_index < len(self.template_names)
            else Config.TEMPLATE_SCREENSHOT_NAMES[0]
        )
        
        # 待ち時間を取得して適用
        delay = self.delays.get(suffix, 0.0)
        if delay > 0:
            time.sleep(delay)
        delay_time = delay
        
        # 撮影回数を取得
        shoot_count = self.shoot_counts.get(suffix, 1)
        
        # 設定回数分だけ撮影（1秒ごと）
        self._take_multiple_screenshots(
            window, suffix, shoot_count, detection_capture_time, 
            detection_time, delay_time
        )
        
        # 撮影完了後、検出状態をリセット
        self.undetected_count = [0] * num_template_groups
    
    def _take_multiple_screenshots(
        self,
        window: gw.Window,
        suffix: str,
        shoot_count: int,
        detection_capture_time: float,
        detection_time: float,
        delay_time: float
    ) -> None:
        """複数回スクリーンショットを撮影（1秒ごと）
        
        Args:
            window: キャプチャ対象のウィンドウオブジェクト
            suffix: ファイル名のサフィックス
            shoot_count: 撮影回数
            detection_capture_time: 検出時のキャプチャ時間
            detection_time: 検出時間
            delay_time: 待ち時間
        """
        for shot_index in range(shoot_count):
            # 待ち時間後の最初の撮影、または前回の撮影から1秒経過後の撮影
            if shot_index > 0:
                time.sleep(Config.SHOOT_INTERVAL)
            
            # 新しいスクリーンショットをキャプチャ
            final_screenshot, final_capture_time, capture_mode = self.window_capturer.capture_window(window)
            if final_screenshot is None:
                self.logger.error(f"撮影{shot_index + 1}回目: スクリーンショットキャプチャに失敗しました")
                continue
            
            # 複数回撮影の場合、ファイル名に連番を追加（1枚目は連番なし）
            shot_number = (shot_index + 1) if shoot_count > 1 else None
            filepath, save_time = self.screenshot_manager.save(final_screenshot, suffix, shot_number)
            
            if filepath and filepath.exists():
                # ログには検出用のキャプチャ時間と保存用のキャプチャ時間の合計を使用
                total_capture_time = detection_capture_time + final_capture_time
                self._log_screenshot_success(
                    filepath, total_capture_time, detection_time, delay_time, save_time, capture_mode,
                    shot_index + 1, shoot_count
                )
                NotificationService.send(f"撮影成功: {filepath.name}", self.logger)
            else:
                self._log_screenshot_failure(filepath)
    
    def _log_screenshot_success(
        self,
        filepath: Path,
        capture_time: float,
        detection_time: float,
        delay_time: float,
        save_time: float,
        capture_mode: str,
        shot_number: int = 1,
        total_shots: int = 1
    ) -> None:
        """スクリーンショット成功をログに記録
        
        Args:
            filepath: 保存先のファイルパス
            capture_time: キャプチャ時間（秒）
            detection_time: 検出時間（秒）
            delay_time: 待ち時間（秒）
            save_time: 保存時間（秒）
            capture_mode: キャプチャモード
            shot_number: 撮影番号（複数回撮影の場合）
            total_shots: 総撮影回数（複数回撮影の場合）
        """
        capture_ms = capture_time * Config.MS_PER_SECOND
        detection_ms = detection_time * Config.MS_PER_SECOND
        delay_ms = delay_time * Config.MS_PER_SECOND
        save_ms = save_time * Config.MS_PER_SECOND
        detection_to_save_ms = (detection_time + delay_time + save_time) * Config.MS_PER_SECOND
        total_ms = (capture_time + detection_time + delay_time + save_time) * Config.MS_PER_SECOND
        
        # 複数回撮影の場合、撮影番号を表示
        shot_info = ""
        if total_shots > 1:
            shot_info = f" | 撮影: {shot_number}/{total_shots}回目"
        
        self.logger.info(
            f"撮影成功: {filepath.name} | "
            f"保存先: {filepath} | "
            f"キャプチャモード: {capture_mode}{shot_info} | "
            f"キャプチャ: {capture_ms:.1f}ms | "
            f"検出: {detection_ms:.1f}ms | "
            f"待ち時間: {delay_ms:.1f}ms | "
            f"保存: {save_ms:.1f}ms | "
            f"検出→保存: {detection_to_save_ms:.1f}ms | "
            f"合計: {total_ms:.1f}ms"
        )
    
    def _log_screenshot_failure(self, filepath: Optional[Path]) -> None:
        """スクリーンショット失敗をログに記録"""
        if filepath:
            self.logger.error(
                f"撮影失敗: ファイルが保存されませんでした | "
                f"パス: {filepath} | "
                f"保存先ディレクトリの存在確認: {self.paths.screenshots_dir.exists()}"
            )
        else:
            self.logger.error("撮影失敗: ファイルパスが取得できませんでした")
    
    def _process_monitoring_cycle(self, last_log_time: float) -> Tuple[float, bool]:
        """1回の監視サイクルを処理"""
        current_time = time.time()
        
        # ログローテーション確認（一定間隔でのみ実行）
        if current_time - self.last_log_rotation_check_time >= Config.LOG_ROTATION_CHECK_INTERVAL:
            self.logger_manager.check_rotation()
            self.last_log_rotation_check_time = current_time
        
        # ウィンドウ検索（キャッシュ機能付き）
        window = self._find_window()
        if window is None:
            error_msg = (
                f"ターゲットウィンドウ '{self.target_window_title}' が見つかりませんでした。\n"
                "アプリケーションを終了します。"
            )
            self.logger.error(error_msg)
            raise RuntimeError(error_msg) from None
        
        # 20秒ごとにログ出力
        if current_time - last_log_time >= Config.LOG_INTERVAL:
            self.logger.info("監視継続中...（終了する場合は Ctrl+C を押すか、このウィンドウを閉じてください）")
            last_log_time = current_time
        
        # 監視間隔チェック
        if not self._should_check_now(current_time):
            return last_log_time, True
        
        self.last_detection_time = current_time
        
        # ウィンドウ処理
        self._process_window_detection(window, current_time)
        
        return last_log_time, True
    
    def run(self) -> None:
        """メインループ"""
        # 起動時にすぐ監視継続中のメッセージを表示
        self.logger.info("監視継続中...（終了する場合は Ctrl+C を押すか、このウィンドウを閉じてください）")
        
        last_log_time = time.time()
        
        try:
            while True:
                last_log_time, _ = self._process_monitoring_cycle(last_log_time)
                time.sleep(Config.SLEEP_INTERVAL)
        except KeyboardInterrupt:
            self.logger.info("アプリケーションを終了します")
        except Exception as e:
            self.logger.error(f"予期しないエラー: {e}")
            raise
        finally:
            # Windows Capture Managerをシャットダウン
            shutdown_capture_manager()

