# -*- coding: utf-8 -*-
"""
設定定数クラスと設定ファイル管理クラス
"""

import logging
import configparser
from pathlib import Path
from typing import List, Optional, Callable, Any


class Config:
    """アプリケーション設定定数クラス"""
    # デフォルト設定
    DEFAULT_WINDOW_TITLE = "FINAL FANTASY XIV"
    DEFAULT_IMAGE_FORMAT = "jpeg"  # jpeg または png
    CONFIG_SECTION = "SETTINGS"
    CONFIG_KEY_IMAGE_FORMAT = "IMAGE_FORMAT"
    CONFIG_KEY_CCRESULT_BASE_SCALE = "CCRESULT_BASE_SCALE"  # ccresultの基準サイズ（デフォルト100）
    CONFIG_KEY_PVPPROFILE_BASE_SCALE = "PVPPROFILE_BASE_SCALE"  # pvpprofileの基準サイズ（デフォルト100）
    CONFIG_KEY_CCRESULT_DELAY = "CCRESULT_DELAY"  # ccresultの待ち時間（デフォルト0秒）
    CONFIG_KEY_PVPPROFILE_DELAY = "PVPPROFILE_DELAY"  # pvpprofileの待ち時間（デフォルト0秒）
    DEFAULT_DELAY = 0.0  # デフォルトの待ち時間（秒）
    CONFIG_KEY_CCRESULT_SHOOT_COUNT = "CCRESULT_SHOOT_COUNT"  # ccresultの撮影回数（デフォルト1回）
    CONFIG_KEY_PVPPROFILE_SHOOT_COUNT = "PVPPROFILE_SHOOT_COUNT"  # pvpprofileの撮影回数（デフォルト1回）
    DEFAULT_SHOOT_COUNT = 1  # デフォルトの撮影回数
    MIN_SHOOT_COUNT = 1  # 最小撮影回数
    MAX_SHOOT_COUNT = 5  # 最大撮影回数
    SHOOT_INTERVAL = 1.0  # 撮影間隔（秒）
    
    # ディレクトリ名
    LOGS_DIR = "logs"  # app/logs（実行時に作成）
    SCREENSHOTS_DIR = "screenshots"  # ccresultss/screenshots（実行時に作成）
    TEMPLATES_DIR = "templates"  # app/templates（プログラムに含まれる）
    
    # テンプレート設定（プログラム内に固定）
    TEMPLATE_BASE_NAMES = ["template1", "template2"]  # テンプレートのベースファイル名
    TEMPLATE_SCREENSHOT_NAMES = ["ccresult", "pvpprofile"]  # スクリーンショット名
    
    # ファイル名
    CONFIG_FILE = "config.ini"
    LOG_FILE_FORMAT = "ccresultss_{date}.log"
    SCREENSHOT_FILE_FORMAT = "{timestamp}_{suffix}"  # 拡張子は動的に追加
    
    # 監視設定
    DETECTION_INTERVAL = 1.0  # 秒（監視間隔）
    LOG_INTERVAL = 20  # 秒（ログ出力間隔）
    SLEEP_INTERVAL = 1.0  # 秒（CPU使用率抑制のための待機時間、検出間隔と同じ）
    WINDOW_CACHE_INTERVAL = 2.0  # 秒（ウィンドウ検索結果のキャッシュ時間）
    LOG_ROTATION_CHECK_INTERVAL = 5.0  # 秒（ログローテーション確認間隔）
    MS_PER_SECOND = 1000  # ミリ秒変換用定数
    
    # テンプレートマッチング
    TEMPLATE_MATCH_THRESHOLD = 0.8  # 検出しきい値
    TEMPLATE_SCALES = [60, 80, 90, 100, 110, 120, 140, 160, 180, 200]  # テンプレート画像のサイズスケール（%）
    DEFAULT_TEMPLATE_BASE_SCALE = 100  # デフォルトの基準サイズ
    MIN_TEMPLATE_SCALE = 60  # 最小テンプレートスケール
    MAX_TEMPLATE_SCALE = 200  # 最大テンプレートスケール
    MAX_SCALES = 3  # 使用する最大スケール数
    
    # 通知設定
    NOTIFICATION_TITLE = "ccresultss"
    NOTIFICATION_TIMEOUT = 5  # 秒（フルスクリーン時は表示されない場合があります）
    
    # ログ設定
    LOG_FORMAT = "[%(asctime)s] %(message)s"
    LOG_TIME_FORMAT = "%H:%M:%S"
    DATE_FORMAT = "%Y%m%d"
    TIMESTAMP_FORMAT = "%Y%m%d_%H%M%S"
    
    # 画像保存設定
    PNG_COMPRESSION_LEVEL = 9  # PNG圧縮レベル（0-9、9が最高圧縮）
    JPEG_QUALITY = 85  # JPEG品質（0-100、高いほど高品質・大容量）


class ConfigManager:
    """設定ファイル管理クラス"""
    
    def __init__(self, config_path: Path):
        """設定マネージャーの初期化
        
        Args:
            config_path: 設定ファイル（config.ini）のパス
        """
        self.config_path = config_path
        self._config: Optional[configparser.ConfigParser] = None
    
    def _get_config(self) -> configparser.ConfigParser:
        """設定ファイルを読み込んで返す（キャッシュあり）
        
        Returns:
            設定ファイルパーサーオブジェクト
        """
        if self._config is None:
            self._config = configparser.ConfigParser()
            self._config.read(self.config_path, encoding='utf-8')
        return self._config
    
    def load_window_title(self) -> str:
        """ウィンドウタイトルを取得（常にFINAL FANTASY XIV）"""
        return Config.DEFAULT_WINDOW_TITLE
    
    def load_image_format(self) -> str:
        """画像フォーマットを読み込み
        
        Returns:
            画像フォーマット文字列（'jpeg' または 'png'）
        """
        try:
            config = self._get_config()
            format_str = config.get(
                Config.CONFIG_SECTION,
                Config.CONFIG_KEY_IMAGE_FORMAT,
                fallback=Config.DEFAULT_IMAGE_FORMAT
            ).lower().strip()
            
            if format_str in ('jpeg', 'jpg', 'png'):
                return 'jpeg' if format_str in ('jpeg', 'jpg') else 'png'
            else:
                logging.warning(f"無効な画像フォーマット: {format_str}、デフォルト ({Config.DEFAULT_IMAGE_FORMAT}) を使用します")
                return Config.DEFAULT_IMAGE_FORMAT
        except Exception as e:
            logging.warning(f"設定ファイルの読み込みに失敗しました: {e}、デフォルト ({Config.DEFAULT_IMAGE_FORMAT}) を使用します")
            return Config.DEFAULT_IMAGE_FORMAT
    
    def load_template_files(self) -> List[str]:
        """テンプレートファイル名リストを取得（プログラム内で固定）"""
        template_files = []
        for base_name in Config.TEMPLATE_BASE_NAMES:
            for scale in Config.TEMPLATE_SCALES:
                template_file = f"{base_name}_{scale}.png"
                template_files.append(template_file)
        return template_files
    
    def calculate_optimal_scales(self, base_scale: int) -> List[int]:
        """基準サイズに最も近いスケールを利用可能なスケールから最大3つ選択
        
        Args:
            base_scale: 基準サイズ（%）
            
        Returns:
            選択されたスケールのリスト（最大3つ）
        """
        available_scales = sorted(Config.TEMPLATE_SCALES)
        base_scale = max(available_scales[0], min(available_scales[-1], base_scale))  # 範囲内に収める
        
        # 基準サイズに最も近い順にソートして、最大3つを選択
        sorted_scales = sorted(available_scales, key=lambda x: abs(x - base_scale))
        return sorted(sorted_scales[:Config.MAX_SCALES])
    
    def _strip_comment(self, value_str: str) -> str:
        """設定値からコメント部分（#以降）を削除
        
        Args:
            value_str: 設定値の文字列
            
        Returns:
            コメントを削除した設定値の文字列
        """
        if '#' in value_str:
            return value_str.split('#')[0].strip()
        return value_str.strip()
    
    def _load_config_value_with_validation(
        self,
        template_name: str,
        config_type: str,
        default_value: Any,
        parser: Callable[[str], Any],
        validator: Callable[[Any], bool],
        error_prefix: str,
        fallback_value: Any
    ) -> Any:
        """設定値を読み込んで検証する汎用メソッド
        
        Args:
            template_name: テンプレート名（"ccresult" または "pvpprofile"）
            config_type: 設定タイプ（"scale", "delay", "shoot_count"）
            default_value: デフォルト値
            parser: 文字列を値に変換する関数（例: int, float）
            validator: 値を検証する関数（boolを返す）
            error_prefix: エラーメッセージのプレフィックス
            fallback_value: 検証失敗時のフォールバック値
            
        Returns:
            設定値（検証済み、またはフォールバック値）
        """
        try:
            config = self._get_config()
            key = self._get_config_key(template_name, config_type)
            value_str = self._strip_comment(
                config.get(
                    Config.CONFIG_SECTION,
                    key,
                    fallback=str(default_value)
                )
            )
            
            try:
                value = parser(value_str)
                if validator(value):
                    return value
                else:
                    logging.warning(
                        f"{error_prefix}: {value}、デフォルト ({fallback_value}) を使用します"
                    )
                    return fallback_value
            except ValueError:
                logging.warning(
                    f"{error_prefix}: {value_str}、デフォルト ({fallback_value}) を使用します"
                )
                return fallback_value
        except Exception as e:
            logging.warning(f"設定ファイルの読み込みに失敗しました: {e}、デフォルト ({fallback_value}) を使用します")
            return fallback_value
    
    def _get_config_key(self, template_name: str, config_type: str) -> str:
        """テンプレート名と設定タイプから設定キーを取得
        
        Args:
            template_name: テンプレート名（"ccresult" または "pvpprofile"）
            config_type: 設定タイプ（"scale", "delay", "shoot_count"）
            
        Returns:
            設定キー文字列
        """
        key_map = {
            "scale": {
                "ccresult": Config.CONFIG_KEY_CCRESULT_BASE_SCALE,
                "pvpprofile": Config.CONFIG_KEY_PVPPROFILE_BASE_SCALE
            },
            "delay": {
                "ccresult": Config.CONFIG_KEY_CCRESULT_DELAY,
                "pvpprofile": Config.CONFIG_KEY_PVPPROFILE_DELAY
            },
            "shoot_count": {
                "ccresult": Config.CONFIG_KEY_CCRESULT_SHOOT_COUNT,
                "pvpprofile": Config.CONFIG_KEY_PVPPROFILE_SHOOT_COUNT
            }
        }
        return key_map.get(config_type, {}).get(template_name, "")
    
    def load_template_base_scale(self, template_name: str) -> int:
        """テンプレートの基準サイズを読み込み（template_nameは"ccresult"または"pvpprofile"）
        
        Args:
            template_name: テンプレート名（"ccresult" または "pvpprofile"）
            
        Returns:
            基準サイズ（整数、60-200の範囲）
        """
        def validate_scale(value: int) -> bool:
            return Config.MIN_TEMPLATE_SCALE <= value <= Config.MAX_TEMPLATE_SCALE
        
        def parse_scale(value_str: str) -> int:
            return int(value_str)
        
        scale = self._load_config_value_with_validation(
            template_name, "scale",
            Config.DEFAULT_TEMPLATE_BASE_SCALE,
            parse_scale, validate_scale,
            f"無効な基準サイズ", Config.DEFAULT_TEMPLATE_BASE_SCALE
        )
        return int(scale)
    
    def load_delay(self, template_name: str) -> float:
        """テンプレートの待ち時間を読み込み（template_nameは"ccresult"または"pvpprofile"）
        
        Args:
            template_name: テンプレート名（"ccresult" または "pvpprofile"）
            
        Returns:
            待ち時間（秒、0以上の浮動小数点数）
        """
        def validate_delay(value: float) -> bool:
            return value >= 0
        
        def parse_delay(value_str: str) -> float:
            return float(value_str)
        
        return self._load_config_value_with_validation(
            template_name, "delay",
            Config.DEFAULT_DELAY,
            parse_delay, validate_delay,
            f"無効な待ち時間", Config.DEFAULT_DELAY
        )
    
    def load_shoot_count(self, template_name: str) -> int:
        """テンプレートの撮影回数を読み込み（template_nameは"ccresult"または"pvpprofile"）
        
        Args:
            template_name: テンプレート名（"ccresult" または "pvpprofile"）
            
            Returns:
            撮影回数（整数、1-5の範囲）
            
        Raises:
            ValueError: 設定値が整数でない場合、または範囲外の場合
        """
        config = self._get_config()
        key = self._get_config_key(template_name, "shoot_count")
        count_str = self._strip_comment(
            config.get(
                Config.CONFIG_SECTION,
                key,
                fallback=str(Config.DEFAULT_SHOOT_COUNT)
            )
        )
        
        try:
            count = int(count_str)
        except ValueError:
            raise ValueError(
                f"設定エラー: {key} の値 '{count_str}' は整数ではありません。"
                f"1から{Config.MAX_SHOOT_COUNT}の整数を指定してください。"
            )
        
        if not (Config.MIN_SHOOT_COUNT <= count <= Config.MAX_SHOOT_COUNT):
            raise ValueError(
                f"設定エラー: {key} の値 {count} は範囲外です。"
                f"{Config.MIN_SHOOT_COUNT}から{Config.MAX_SHOOT_COUNT}の範囲で指定してください。"
            )
        
        return count
