# -*- coding: utf-8 -*-
"""
Windows Graphics Capture API (WinRT) を使ったウィンドウキャプチャ実装
windows-capture ライブラリを使用して実装
Windows 11対応、draw_border=Falseで黄色い境界線を非表示
"""

import logging
import threading
from typing import Optional
import numpy as np
import pygetwindow as gw

# windows-capture ライブラリ
try:
    from windows_capture import WindowsCapture, Frame, InternalCaptureControl
    WINDOWS_CAPTURE_AVAILABLE = True
except ImportError:
    WINDOWS_CAPTURE_AVAILABLE = False
    # 型チェッカーのためにダミークラスを定義
    WindowsCapture = None
    Frame = None
    InternalCaptureControl = None


class WindowsCaptureManager:
    """Windows Capture API を使用したウィンドウキャプチャマネージャー
    
    最新フレームを保持し、メインループから取得できるようにします。
    """
    
    def __init__(self, window_name: str, logger: logging.Logger):
        """初期化
        
        Args:
            window_name: キャプチャ対象のウィンドウ名
            logger: ロガー
        """
        self.window_name = window_name
        self.logger = logger
        self.latest_frame: Optional[np.ndarray] = None
        self.frame_lock = threading.Lock()
        self.capture: Optional[WindowsCapture] = None
        self.capture_control: Optional[InternalCaptureControl] = None
        self.is_running = False
        
    def start_capture(self) -> bool:
        """キャプチャを開始
        
        Returns:
            成功した場合True、失敗した場合False
        """
        if not WINDOWS_CAPTURE_AVAILABLE:
            self.logger.debug("windows-captureライブラリが利用できません")
            return False
        
        try:
            # WindowsCapture インスタンスを作成
            # draw_border=False で黄色い境界線を非表示
            # cursor_capture=False でカーソルをキャプチャしない
            self.capture = WindowsCapture(
                window_name=self.window_name,
                draw_border=False,
                cursor_capture=False
            )
            
            # イベントハンドラを登録
            @self.capture.event
            def on_frame_arrived(frame: Frame, capture_control: InternalCaptureControl) -> None:
                """フレームが到着したときのイベントハンドラ"""
                self._on_frame_arrived(frame, capture_control)
            
            @self.capture.event
            def on_closed() -> None:
                """キャプチャが閉じられたときのイベントハンドラ"""
                self._on_closed()
            
            # バックグラウンドスレッドでキャプチャを開始
            self.capture_control = self.capture.start_free_threaded()
            self.is_running = True
            self.logger.info(f"Windows Capture API (WinRT) でキャプチャを開始しました: {self.window_name}")
            return True
            
        except Exception as e:
            self.logger.warning(f"Windows Capture API (WinRT) の初期化に失敗: {e}")
            self.capture = None
            self.capture_control = None
            self.is_running = False
            return False
    
    def stop_capture(self) -> None:
        """キャプチャを停止"""
        if self.capture_control is not None and self.is_running:
            try:
                self.capture_control.stop()
            except Exception as e:
                self.logger.warning(f"Windows Capture API (WinRT) の停止中にエラー: {e}")
            finally:
                self.is_running = False
                self.capture_control = None
                self.capture = None
        
        # 最新フレームをクリア
        with self.frame_lock:
            self.latest_frame = None
    
    def _on_frame_arrived(self, frame: Frame, capture_control: InternalCaptureControl) -> None:
        """フレームが到着したときのコールバック
        
        Args:
            frame: windows-captureから提供されるフレームオブジェクト
            capture_control: キャプチャコントロールオブジェクト
        """
        try:
            # BGR形式に変換
            bgr_frame = frame.convert_to_bgr()
            
            # frame_buffer属性からnumpy配列を取得
            if hasattr(bgr_frame, 'frame_buffer'):
                frame_array = bgr_frame.frame_buffer
                
                # numpy配列であることを確認
                if isinstance(frame_array, np.ndarray):
                    # 最新フレームを更新（スレッドセーフ）
                    with self.frame_lock:
                        self.latest_frame = frame_array.copy()
                else:
                    self.logger.debug(f"frame_bufferがnumpy配列ではありません: {type(frame_array)}")
            else:
                self.logger.debug("BGRフレームにframe_buffer属性がありません")
                
        except Exception as e:
            self.logger.debug(f"フレーム変換エラー: {e}")
    
    def _on_closed(self) -> None:
        """キャプチャが閉じられたときのコールバック"""
        self.logger.info("Windows Capture API (WinRT) のキャプチャが閉じられました")
        self.is_running = False
        
        # 最新フレームをクリア
        with self.frame_lock:
            self.latest_frame = None
    
    def get_latest_frame(self) -> Optional[np.ndarray]:
        """最新のフレームを取得
        
        Returns:
            最新のフレーム（BGR形式のnumpy配列）、取得できない場合None
        """
        with self.frame_lock:
            if self.latest_frame is not None:
                return self.latest_frame.copy()
            return None


# グローバルなキャプチャマネージャー（シングルトン）
_capture_manager: Optional[WindowsCaptureManager] = None
_manager_lock = threading.Lock()


def initialize_capture_manager(window_name: str, logger: logging.Logger) -> bool:
    """キャプチャマネージャーを初期化
    
    Args:
        window_name: キャプチャ対象のウィンドウ名
        logger: ロガー
        
    Returns:
        初期化に成功した場合True、失敗した場合False
    """
    global _capture_manager
    
    with _manager_lock:
        if _capture_manager is not None:
            # 既に初期化されている場合は停止してから再初期化
            _capture_manager.stop_capture()
        
        _capture_manager = WindowsCaptureManager(window_name, logger)
        return _capture_manager.start_capture()


def shutdown_capture_manager() -> None:
    """キャプチャマネージャーをシャットダウン"""
    global _capture_manager
    
    with _manager_lock:
        if _capture_manager is not None:
            _capture_manager.stop_capture()
            _capture_manager = None


def capture_window_with_winrt(window: gw.Window, logger: logging.Logger) -> Optional[np.ndarray]:
    """Windows Graphics Capture API (WinRT) を使ってウィンドウの内容をキャプチャ
    
    Args:
        window: キャプチャ対象のウィンドウオブジェクト（互換性のため保持、実際には使用しない）
        logger: ロガー
        
    Returns:
        キャプチャした画像データ（BGR形式）、失敗時はNone
        
    注意:
        windows-captureライブラリを使用して、Windows Graphics Capture APIを
        利用します。最新フレームをグローバルなキャプチャマネージャーから取得します。
    """
    global _capture_manager
    
    if not WINDOWS_CAPTURE_AVAILABLE:
        logger.debug("windows-captureライブラリが利用できません（未インストール）")
        return None
    
    if _capture_manager is None:
        logger.debug("Windows Capture Managerが初期化されていません")
        return None
    
    try:
        # 最新フレームを取得
        frame = _capture_manager.get_latest_frame()
        if frame is not None:
            return frame
        else:
            logger.debug("最新フレームが取得できませんでした（まだフレームが到着していない可能性があります）")
            return None
            
    except Exception as e:
        logger.debug(f"WinRT Graphics Captureエラー: {e}")
        return None
