# -*- coding: utf-8 -*-
"""
Windows API（PrintWindow）を使ったウィンドウキャプチャ実装
"""

import logging
from typing import Optional
import numpy as np
import pygetwindow as gw

# Windows API用のctypes
try:
    import ctypes
    from ctypes import wintypes
    WINDOWS_AVAILABLE = True
except ImportError:
    WINDOWS_AVAILABLE = False

if WINDOWS_AVAILABLE:
    # Windows API定数
    PW_CLIENTONLY = 0x1
    DIB_RGB_COLORS = 0
    BI_RGB = 0
    RGB_BIT_COUNT = 24  # 24bit RGB
    RGB_PLANES = 1
    
    # Windows API DLL
    user32 = ctypes.windll.user32
    gdi32 = ctypes.windll.gdi32
    
    # BITMAPINFO構造体（GetDIBitsの型定義より前に定義する必要がある）
    class BITMAPINFOHEADER(ctypes.Structure):
        """BITMAPINFOHEADER構造体"""
        _fields_ = [
            ("biSize", ctypes.c_uint32),
            ("biWidth", ctypes.c_int32),
            ("biHeight", ctypes.c_int32),
            ("biPlanes", ctypes.c_uint16),
            ("biBitCount", ctypes.c_uint16),
            ("biCompression", ctypes.c_uint32),
            ("biSizeImage", ctypes.c_uint32),
            ("biXPelsPerMeter", ctypes.c_int32),
            ("biYPelsPerMeter", ctypes.c_int32),
            ("biClrUsed", ctypes.c_uint32),
            ("biClrImportant", ctypes.c_uint32),
        ]
    
    class BITMAPINFO(ctypes.Structure):
        """BITMAPINFO構造体"""
        _fields_ = [
            ("bmiHeader", BITMAPINFOHEADER),
            ("bmiColors", ctypes.c_uint32 * 1),  # RGBQUAD配列（1要素のみ）
        ]
    
    # Windows API関数の定義
    PrintWindow = user32.PrintWindow
    PrintWindow.argtypes = [wintypes.HWND, wintypes.HDC, ctypes.c_uint]
    PrintWindow.restype = wintypes.BOOL
    
    GetWindowDC = user32.GetWindowDC
    GetWindowDC.argtypes = [wintypes.HWND]
    GetWindowDC.restype = wintypes.HDC
    
    CreateCompatibleDC = gdi32.CreateCompatibleDC
    CreateCompatibleDC.argtypes = [wintypes.HDC]
    CreateCompatibleDC.restype = wintypes.HDC
    
    CreateCompatibleBitmap = gdi32.CreateCompatibleBitmap
    CreateCompatibleBitmap.argtypes = [wintypes.HDC, ctypes.c_int, ctypes.c_int]
    CreateCompatibleBitmap.restype = wintypes.HBITMAP
    
    SelectObject = gdi32.SelectObject
    SelectObject.argtypes = [wintypes.HDC, wintypes.HGDIOBJ]
    SelectObject.restype = wintypes.HGDIOBJ
    
    DeleteObject = gdi32.DeleteObject
    DeleteObject.argtypes = [wintypes.HGDIOBJ]
    DeleteObject.restype = wintypes.BOOL
    
    DeleteDC = gdi32.DeleteDC
    DeleteDC.argtypes = [wintypes.HDC]
    DeleteDC.restype = wintypes.BOOL
    
    ReleaseDC = user32.ReleaseDC
    ReleaseDC.argtypes = [wintypes.HWND, wintypes.HDC]
    ReleaseDC.restype = ctypes.c_int
    
    GetDIBits = gdi32.GetDIBits
    GetDIBits.argtypes = [
        wintypes.HDC,
        wintypes.HBITMAP,
        ctypes.c_uint,
        ctypes.c_uint,
        ctypes.POINTER(ctypes.c_ubyte),
        ctypes.POINTER(BITMAPINFO),
        ctypes.c_uint,
    ]
    GetDIBits.restype = ctypes.c_int


def get_window_handle(window: gw.Window) -> Optional[int]:
    """pygetwindowのWindowオブジェクトからウィンドウハンドル（HWND）を取得
    
    Args:
        window: pygetwindowのWindowオブジェクト
        
    Returns:
        ウィンドウハンドル（HWND）、取得できない場合はNone
    """
    try:
        # pygetwindowのWindowオブジェクトには_hWnd属性がある
        if hasattr(window, '_hWnd'):
            return window._hWnd
        return None
    except Exception:
        return None


def _create_bitmap_info(width: int, height: int) -> BITMAPINFO:
    """BITMAPINFO構造体を作成
    
    Args:
        width: ビットマップの幅
        height: ビットマップの高さ
        
    Returns:
        BITMAPINFO構造体インスタンス
    """
    if not WINDOWS_AVAILABLE:
        raise RuntimeError("Windows API is not available")
    
    bmi = BITMAPINFO()
    bmi.bmiHeader.biSize = ctypes.sizeof(BITMAPINFOHEADER)
    bmi.bmiHeader.biWidth = width
    bmi.bmiHeader.biHeight = -height  # 負の値でトップダウン形式
    bmi.bmiHeader.biPlanes = RGB_PLANES
    bmi.bmiHeader.biBitCount = RGB_BIT_COUNT
    bmi.bmiHeader.biCompression = BI_RGB
    return bmi


def _render_window_to_bitmap(hwnd: ctypes.c_void_p, hdc: wintypes.HDC) -> bool:
    """PrintWindowを使ってウィンドウの内容をビットマップに描画
    
    Args:
        hwnd: ウィンドウハンドル
        hdc: デバイスコンテキスト
        
    Returns:
        成功時True、失敗時False
    """
    # まずPW_CLIENTONLYフラグで試行
    result = PrintWindow(hwnd, hdc, PW_CLIENTONLY)
    if result:
        return True
    
    # PW_CLIENTONLYが失敗した場合、フラグなしで再試行
    result = PrintWindow(hwnd, hdc, 0)
    return bool(result)


def _extract_bitmap_data(
    hdc: wintypes.HDC,
    hbitmap: wintypes.HBITMAP,
    width: int,
    height: int
) -> Optional[np.ndarray]:
    """ビットマップから画像データを取得
    
    Args:
        hdc: デバイスコンテキスト
        hbitmap: ビットマップハンドル
        width: 画像の幅
        height: 画像の高さ
        
    Returns:
        画像データ（BGR形式）、失敗時はNone
    """
    # BITMAPINFO構造体を準備
    bmi = _create_bitmap_info(width, height)
    
    # 画像データを格納する配列を作成
    array = np.zeros((height, width, 3), dtype=np.uint8)
    
    # GetDIBitsでビットマップのデータを取得
    lines = GetDIBits(
        hdc,
        hbitmap,
        0,
        height,
        array.ctypes.data_as(ctypes.POINTER(ctypes.c_ubyte)),
        ctypes.byref(bmi),
        DIB_RGB_COLORS,
    )
    
    if lines > 0:
        return array  # BGR形式（WindowsはBGRを返す）
    return None


def _setup_capture_resources(
    hwnd: ctypes.c_void_p, width: int, height: int
) -> Optional[tuple]:
    """キャプチャに必要なリソース（DC、ビットマップ）をセットアップ
    
    Args:
        hwnd: ウィンドウハンドル
        width: ビットマップの幅
        height: ビットマップの高さ
        
    Returns:
        (hdc_src, hdc_dest, hbitmap, old_bitmap) のタプル、失敗時はNone
    """
    hdc_src = GetWindowDC(hwnd)
    if not hdc_src:
        return None
    
    hdc_dest = CreateCompatibleDC(hdc_src)
    if not hdc_dest:
        ReleaseDC(hwnd, hdc_src)
        return None
    
    hbitmap = CreateCompatibleBitmap(hdc_src, width, height)
    if not hbitmap:
        DeleteDC(hdc_dest)
        ReleaseDC(hwnd, hdc_src)
        return None
    
    old_bitmap = SelectObject(hdc_dest, hbitmap)
    return (hdc_src, hdc_dest, hbitmap, old_bitmap)


def _cleanup_capture_resources(
    hwnd: ctypes.c_void_p,
    hdc_src: wintypes.HDC,
    hdc_dest: wintypes.HDC,
    hbitmap: wintypes.HBITMAP,
    old_bitmap: wintypes.HGDIOBJ
) -> None:
    """キャプチャリソースをクリーンアップ
    
    Args:
        hwnd: ウィンドウハンドル
        hdc_src: ソースデバイスコンテキスト
        hdc_dest: デスティネーションデバイスコンテキスト
        hbitmap: ビットマップハンドル
        old_bitmap: 以前のビットマップハンドル
    """
    SelectObject(hdc_dest, old_bitmap)
    DeleteObject(hbitmap)
    DeleteDC(hdc_dest)
    ReleaseDC(hwnd, hdc_src)


def capture_window_with_printwindow(window: gw.Window, logger: logging.Logger) -> Optional[np.ndarray]:
    """PrintWindow APIを使ってウィンドウの内容をキャプチャ
    
    Args:
        window: キャプチャ対象のウィンドウオブジェクト
        logger: ロガー（未使用だが、インターフェースの一貫性のため保持）
        
    Returns:
        キャプチャした画像データ（BGR形式）、失敗時はNone
    """
    if not WINDOWS_AVAILABLE:
        return None
    
    hwnd_value = get_window_handle(window)
    if hwnd_value is None:
        return None
    
    hwnd = ctypes.c_void_p(hwnd_value)
    width = window.width
    height = window.height
    
    # キャプチャリソースをセットアップ
    resources = _setup_capture_resources(hwnd, width, height)
    if resources is None:
        return None
    
    hdc_src, hdc_dest, hbitmap, old_bitmap = resources
    
    try:
        # PrintWindowでウィンドウの内容を描画
        if not _render_window_to_bitmap(hwnd, hdc_dest):
            return None
        
        # ビットマップから画像データを取得
        return _extract_bitmap_data(hdc_dest, hbitmap, width, height)
    finally:
        _cleanup_capture_resources(hwnd, hdc_src, hdc_dest, hbitmap, old_bitmap)

