# -*- coding: utf-8 -*-
"""
ウィンドウキャプチャクラスとスクリーンショット管理クラス
"""

import time
import logging
from datetime import datetime
from pathlib import Path
from typing import Optional, Tuple, List, Any, Callable
import cv2
import numpy as np
import mss
import pygetwindow as gw
from config import Config

# キャプチャモード定数
CAPTURE_MODE_WINRT = "WinRT"
CAPTURE_MODE_PRINTWINDOW = "PrintWindow"
CAPTURE_MODE_MSS = "mss"

# Windows Graphics Capture API (WinRT) 実装をインポート（優先、windows-captureライブラリを使用）
try:
    from capture_winrt import capture_window_with_winrt, WINDOWS_CAPTURE_AVAILABLE
except ImportError:
    WINDOWS_CAPTURE_AVAILABLE = False
    def capture_window_with_winrt(window: gw.Window, logger: logging.Logger) -> Optional[np.ndarray]:
        return None

# PrintWindow実装をインポート（Windows APIを使う場合、WinRTが失敗した場合のフォールバック）
try:
    from capture_printwindow import capture_window_with_printwindow, WINDOWS_AVAILABLE
except ImportError:
    WINDOWS_AVAILABLE = False
    def capture_window_with_printwindow(window: gw.Window, logger: logging.Logger) -> Optional[np.ndarray]:
        return None


class WindowCapturer:
    """ウィンドウキャプチャクラス（WinRT Graphics Capture APIを優先、PrintWindow、mssの順にフォールバック）"""
    
    def __init__(self, logger: logging.Logger):
        """ウィンドウキャプチャの初期化"""
        self.logger = logger
        self.mss_instance = mss.mss()
        self.use_winrt = WINDOWS_CAPTURE_AVAILABLE
        self.use_printwindow = WINDOWS_AVAILABLE
    
    def _create_monitor_dict(self, window: gw.Window) -> dict:
        """ウィンドウ情報からmss用のmonitor辞書を作成
        
        Args:
            window: キャプチャ対象のウィンドウオブジェクト
            
        Returns:
            mss用のmonitor辞書
        """
        return {
            "top": window.top,
            "left": window.left,
            "width": window.width,
            "height": window.height
        }
    
    def _convert_screenshot_to_bgr(self, screenshot: Any) -> np.ndarray:
        """mssのスクリーンショットをBGR形式のnumpy配列に変換
        
        Args:
            screenshot: mssのgrab()の戻り値
            
        Returns:
            BGR形式のnumpy配列
        """
        # PILを使わずに直接numpy配列に変換（高速化）
        img_np = np.frombuffer(screenshot.rgb, dtype=np.uint8).reshape(
            (screenshot.height, screenshot.width, 3)
        )
        # BGRに変換（OpenCVはBGRを期待するため）
        return cv2.cvtColor(img_np, cv2.COLOR_RGB2BGR)
    
    def _capture_with_mss(self, window: gw.Window) -> Tuple[Optional[np.ndarray], float, str]:
        """mssライブラリを使ってウィンドウをキャプチャ
        
        Args:
            window: キャプチャ対象のウィンドウオブジェクト
            
        Returns:
            タプル (キャプチャした画像データ（BGR形式）、処理時間（秒）、キャプチャモード)
        """
        start_time = time.perf_counter()
        try:
            monitor = self._create_monitor_dict(window)
            screenshot = self.mss_instance.grab(monitor)
            img_bgr = self._convert_screenshot_to_bgr(screenshot)
            elapsed_time = time.perf_counter() - start_time
            return img_bgr, elapsed_time, CAPTURE_MODE_MSS
        except Exception as e:
            self.logger.error(f"mssキャプチャエラー: {e}")
            elapsed_time = time.perf_counter() - start_time
            return None, elapsed_time, CAPTURE_MODE_MSS
    
    def capture_window(self, window: gw.Window) -> Tuple[Optional[np.ndarray], float, str]:
        """ウィンドウ全体をキャプチャし、処理時間（秒）とキャプチャモードも返す
        
        Windows Graphics Capture API (WinRT) を最優先で使用し、失敗した場合は
        PrintWindow API、さらに失敗した場合はmssにフォールバックします。
        WinRT/PrintWindowを使用すると、他のウィンドウに隠れていてもウィンドウの内容をキャプチャできます。
        
        Args:
            window: キャプチャ対象のウィンドウオブジェクト
            
        Returns:
            タプル (キャプチャした画像データ（BGR形式）、処理時間（秒）、キャプチャモード)
        """
        start_time = time.perf_counter()
        
        # WinRT Graphics Capture APIを最優先で試行
        if self.use_winrt:
            image = self._try_winrt_capture(window)
            if image is not None:
                elapsed_time = time.perf_counter() - start_time
                return image, elapsed_time, CAPTURE_MODE_WINRT
        
        # PrintWindowを試行（WinRTが失敗した場合）
        if self.use_printwindow:
            image = self._try_printwindow_capture(window)
            if image is not None:
                elapsed_time = time.perf_counter() - start_time
                return image, elapsed_time, CAPTURE_MODE_PRINTWINDOW
        
        # WinRT/PrintWindowが失敗または利用不可の場合、mssにフォールバック
        return self._capture_with_mss(window)
    
    def _try_capture_method(
        self,
        window: gw.Window,
        capture_func: Callable[[gw.Window, logging.Logger], Optional[np.ndarray]],
        method_name: str,
        fallback_message: str,
        log_level: str = "debug"
    ) -> Optional[np.ndarray]:
        """キャプチャメソッドを試行する汎用メソッド
        
        Args:
            window: キャプチャ対象のウィンドウオブジェクト
            capture_func: キャプチャ関数
            method_name: メソッド名（ログ用）
            fallback_message: フォールバック時のメッセージ
            log_level: ログレベル（"debug" または "warning"）
            
        Returns:
            キャプチャした画像データ、失敗時はNone
        """
        try:
            image = capture_func(window, self.logger)
            if image is None:
                log_msg = f"{method_name}が失敗しました（Noneが返されました）、{fallback_message}"
                if log_level == "warning":
                    self.logger.warning(log_msg)
                else:
                    self.logger.debug(log_msg)
            return image
        except Exception as e:
            log_msg = f"{method_name}エラー: {e}、{fallback_message}"
            if log_level == "warning":
                self.logger.warning(log_msg)
            else:
                self.logger.debug(log_msg)
            return None
    
    def _try_winrt_capture(self, window: gw.Window) -> Optional[np.ndarray]:
        """WinRT Graphics Capture APIでのキャプチャを試行
        
        Args:
            window: キャプチャ対象のウィンドウオブジェクト
            
        Returns:
            キャプチャした画像データ、失敗時はNone
        """
        return self._try_capture_method(
            window,
            capture_window_with_winrt,
            "WinRT Graphics Capture",
            "PrintWindowにフォールバックします",
            "debug"
        )
    
    def _try_printwindow_capture(self, window: gw.Window) -> Optional[np.ndarray]:
        """PrintWindowでのキャプチャを試行
        
        Args:
            window: キャプチャ対象のウィンドウオブジェクト
            
        Returns:
            キャプチャした画像データ、失敗時はNone
        """
        return self._try_capture_method(
            window,
            capture_window_with_printwindow,
            "PrintWindowキャプチャ",
            "mssにフォールバックします",
            "warning"
        )


class ScreenshotManager:
    """スクリーンショット管理クラス"""
    
    def __init__(self, screenshots_dir: Path, logger: logging.Logger, image_format: str = "jpeg"):
        """スクリーンショットマネージャーの初期化"""
        self.screenshots_dir = screenshots_dir
        self.logger = logger
        # jpeg/jpgをjpegに統一、それ以外はpng
        self.image_format = "jpeg" if image_format.lower() in ("jpeg", "jpg") else "png"
    
    def _get_encode_params(self) -> List[int]:
        """画像フォーマットに応じたエンコードパラメータを取得"""
        if self.image_format == "jpeg":
            return [cv2.IMWRITE_JPEG_QUALITY, Config.JPEG_QUALITY]
        return [cv2.IMWRITE_PNG_COMPRESSION, Config.PNG_COMPRESSION_LEVEL]
    
    def _get_filepath(self, suffix: str, shot_number: Optional[int] = None) -> Path:
        """保存先のファイルパスを生成
        
        Args:
            suffix: ファイル名のサフィックス（例: "ccresult", "pvpprofile"）
            shot_number: 撮影番号（複数回撮影の場合、Noneでない場合はファイル名に連番を追加）
            
        Returns:
            ファイルパス
        """
        timestamp = datetime.now().strftime(Config.TIMESTAMP_FORMAT)
        base_filename = Config.SCREENSHOT_FILE_FORMAT.format(timestamp=timestamp, suffix=suffix)
        
        # 複数回撮影の場合、ファイル名に連番を追加
        if shot_number is not None and shot_number > 1:
            base_filename = f"{base_filename}_{shot_number}"
        
        extension = ".jpg" if self.image_format == "jpeg" else ".png"
        return self.screenshots_dir / f"{base_filename}{extension}"
    
    def _encode_image(self, image: np.ndarray, extension: str) -> Optional[bytes]:
        """画像をエンコード
        
        Args:
            image: エンコードする画像
            extension: ファイル拡張子（".jpg" または ".png"）
            
        Returns:
            エンコードされた画像データ、失敗時はNone
        """
        encode_params = self._get_encode_params()
        success, encoded_image = cv2.imencode(extension, image, encode_params)
        
        if not success:
            return None
        
        return encoded_image.tobytes()
    
    def _write_image_file(self, filepath: Path, image_data: bytes) -> bool:
        """画像ファイルを書き込む
        
        Args:
            filepath: 保存先のファイルパス
            image_data: 書き込む画像データ
            
        Returns:
            成功時True、失敗時False
        """
        try:
            with open(filepath, 'wb') as f:
                f.write(image_data)
            return filepath.exists()
        except Exception as e:
            self.logger.error(f"ファイル書き込みエラー: {filepath} ({e})")
            return False
    
    def save(self, image: np.ndarray, suffix: str = "ccresult", shot_number: Optional[int] = None) -> Tuple[Optional[Path], float]:
        """スクリーンショットを保存し、処理時間（秒）も返す
        
        Args:
            image: 保存する画像
            suffix: ファイル名のサフィックス（例: "ccresult", "pvpprofile"）
            shot_number: 撮影番号（複数回撮影の場合、Noneでない場合はファイル名に連番を追加）
            
        Returns:
            タプル (保存先のファイルパス、処理時間（秒）)
        """
        start_time = time.perf_counter()
        
        try:
            filepath = self._get_filepath(suffix, shot_number)
            extension = filepath.suffix
            
            # 日本語パス対応: cv2.imencode + バイナリ書き込みを使用
            # cv2.imwriteは日本語パスを正しく処理できない場合があるため
            image_data = self._encode_image(image, extension)
            if image_data is None:
                self.logger.error(f"画像のエンコードに失敗しました: {filepath}")
                elapsed_time = time.perf_counter() - start_time
                return None, elapsed_time
            
            if not self._write_image_file(filepath, image_data):
                elapsed_time = time.perf_counter() - start_time
                return None, elapsed_time
            
            elapsed_time = time.perf_counter() - start_time
            return filepath, elapsed_time
        except Exception as e:
            self.logger.error(f"スクリーンショット保存エラー: {e}")
            elapsed_time = time.perf_counter() - start_time
            return None, elapsed_time

